package com.example.demo_image.service;

import com.example.demo_image.dto.PageResult;
import com.example.demo_image.entity.sys.FileEntity;
import com.example.demo_image.repository.FileRepository;
import com.example.demo_image.util.FileUtils;
import com.example.demo_image.util.SnowflakeIdWorker;
import com.example.demo_image.vo.sys.FileVo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.transaction.Transactional;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author morningbear
 * @date 20.7.13
 */
@Slf4j
@Service
public class FileStoreService {

    @Resource
    private FileRepository fileRepository;

    @Resource
    private SnowflakeIdWorker snowflakeIdWorker;

    private PageResult<FileVo> getFileVoPageResult(Integer pageSize, Integer pageNumber, Page<FileEntity> page) throws IOException{
        List<FileVo> fileVoList = new ArrayList<>();
        for(FileEntity fileEntity : page.getContent()){
            log.info("fileId->{}", fileEntity.getId());
            FileVo fileVo = new FileVo();
            BeanUtils.copyProperties(fileEntity, fileVo);
            fileVo.setId(fileEntity.getId());
            //Path path = Paths.get(fileEntity.getFilePath(), fileEntity.getFileName());
            //log.info("load image from " + path.toString());
            //fileVo.setContent(Files.newInputStream(path));
            fileVoList.add(fileVo);
        }

        return new PageResult<>(pageNumber, pageSize, page.getTotalPages(), fileVoList);
    }

    /**
     * 一些基本文件基本属性
     * @param fileEntity 文件实体
     */
    private void fileInitial(FileEntity fileEntity) {
        String id = String.valueOf(snowflakeIdWorker.nextId());
        fileEntity.setId(id);
        fileEntity.setCreateAt(new Date());
        fileEntity.setCreateBy(1L);
        fileEntity.setUpdateAt(new Date());
        fileEntity.setUpdateBy(1L);
    }

    /**
     * 返回所有图片的信息
     * @return PageResult<FileVo>
     */
    public PageResult<FileVo> findAllFiles(Integer pageNumber, Integer pageSize) throws IOException{
        Page<FileEntity> page = fileRepository.findAll(PageRequest.of(pageNumber, pageSize));
        return getFileVoPageResult(pageSize, pageNumber, page);
    }

    /**
     * 按ID查找图片
     * @return FileVo
     * 应该加一个检测找到的文件是不是空文件
     */
    public FileVo findFileById(String id) throws IOException{
        if (id == null){
            return null;
        }else {
            log.info("id->{}", id);
            FileEntity fileEntity = fileRepository.findFileEntityById(id);
            FileVo fileVo = new FileVo();
            BeanUtils.copyProperties(fileEntity, fileVo);
            Path path = Paths.get(fileEntity.getFilePath(), fileEntity.getFileName());
            fileVo.setContent(Files.newInputStream(path));
            return fileVo;
        }
    }

    /**
     * 按文件名查找
     * @param fileName 文件名
     * @return 图片
     */
    public FileVo findFileByFileName(String fileName) throws IOException{
        if(fileName == null){
            return null;
        }else {
            log.info("fileName->{}", fileName);
            FileEntity fileEntity = fileRepository.findByFileName(fileName);
            FileVo fileVo = new FileVo();
            BeanUtils.copyProperties(fileEntity, fileVo);
            Path path = Paths.get(fileEntity.getFilePath(), fileEntity.getFileName());
            log.info("path->{}", path);
            fileVo.setContent(Files.newInputStream(path));
            return fileVo;

        }
    }

    /**
     * 添加图片。这个只保存图片信息，仅做测试
     * @return 文件id
     */
    @Transactional(rollbackOn = Exception.class)
    public String addFile(FileVo fileVo){
        FileEntity fileEntity = new FileEntity();
        BeanUtils.copyProperties(fileVo, fileEntity);
        fileInitial(fileEntity);
        fileRepository.save(fileEntity);
        return fileEntity.getId();
    }

    /**
     * 上传后存盘
     * @param file 文件
     * @return 文件id
     */
    @Transactional(rollbackOn = Exception.class)
    public String addFileAndStore(MultipartFile file){
        //基础路径
        String basePath = "E:/xi_an";
        //文件保存路径
        String folder = FileUtils.getFolder();
        String fileName = FileUtils.getFileName() + FileUtils.getFileNameSub(file.getOriginalFilename());
        FileEntity fileEntity = new FileEntity();
        fileInitial(fileEntity);
        try {
            Path filePath = Files.createDirectories(Paths.get(basePath, folder));
            log.info("path->{}", filePath);
            Path fullPath = Paths.get(basePath, folder, fileName);
            log.info("fullPath->{}", fullPath);
            Files.write(fullPath, file.getBytes(), StandardOpenOption.CREATE);

            //文件信息
            fileEntity.setFileOriginName(file.getOriginalFilename());
            fileEntity.setFileType(file.getContentType());
            fileEntity.setSize(file.getSize());
            fileEntity.setFileName(fileName);
            fileEntity.setFilePath(filePath.toString());

            fileRepository.save(fileEntity);
        }catch (IOException e){
            e.printStackTrace();
        }
        return fileEntity.getId();
    }

    /**
     @Value("${file.upload.path}")
     private String file;

     private final Path filePath = Paths.get(file);

     public Stream<Path> loadAll(){
     try {
     return Files.walk(filePath, 1)
     .filter(path -> !path.equals(filePath))
     .map(filePath::relativize);
     }
     catch (IOException e){
     throw new RuntimeException(e);
     }
     }
     */
}
