package com.example.demo_image.controller.api;

import com.example.demo_image.controller.BaseController;
import com.example.demo_image.dto.PageResult;
import com.example.demo_image.dto.Result;
import com.example.demo_image.service.FileStoreService;
import com.example.demo_image.vo.sys.FileVo;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.InputStreamResource;
import org.springframework.core.io.InputStreamSource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.IOException;

/**
 * @author morningbear
 * @date 20.7.13
 */
@Slf4j
@RestController
@ApiModel("处理图片上传的controller层接口")
public class FileController extends BaseController {

    @Resource
    private FileStoreService fileStoreService;

    /**
     * 图片文件上传
     * @param file 图片文件
     * @return HttpStatus
     */
    @ApiOperation("图片文件上传")
    @ApiImplicitParam(name = "file", value = "图片文件", required = true, dataType = "__file")
    @PostMapping(value = "/upload", consumes = "multipart/*", headers = "content-type=multipart/form-data")
    public ResponseEntity<Result> addFile(@RequestParam(value = "file") MultipartFile file){
        if (file.isEmpty()){
            return new ResponseEntity<>(new Result().serverError("文件不存在，请重新上传"), HttpStatus.NO_CONTENT);
        }
        fileStoreService.addFileAndStore(file);
        return new ResponseEntity<>(new Result().ok("add success"), HttpStatus.OK);
    }

    /**
     * 按文件名查询图片
     * @param fileName 文件名
     * @return 图片
     */
    @ApiOperation("按文件名查询图片")
    @ApiImplicitParam(name = "fileName", value = "文件名")
    @GetMapping(value = "/file/fileName/{fileName}", produces = MediaType.IMAGE_JPEG_VALUE)
    public ResponseEntity<InputStreamSource> findFileByFileName(@PathVariable String fileName){
        FileVo fileVo = null;
        try {
            fileVo = fileStoreService.findFileByFileName(fileName);
        }catch (IOException e){
            e.printStackTrace();
        }
        if (fileVo == null){
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }
        return new ResponseEntity<>(new InputStreamResource(fileVo.getContent()), HttpStatus.OK);
    }
    @ApiOperation("按id查询图片")
    @ApiImplicitParam(name = "id", value = "编号")
    @GetMapping(value = "/file/ID/{id}", produces = MediaType.IMAGE_JPEG_VALUE)
    public ResponseEntity<InputStreamResource> findFileById(@PathVariable String id){
        FileVo fileVo = null;
        try {
            fileVo = fileStoreService.findFileById(id);
        }catch (IOException e){
            e.printStackTrace();
        }
        if (fileVo == null){
            return new ResponseEntity<>(HttpStatus.NOT_FOUND);
        }
        return new ResponseEntity<>(new InputStreamResource(fileVo.getContent()), HttpStatus.OK);
    }

    /**
     * 查询所有图片文件
     */
    @ApiOperation("查询所有图片文件并返回详细信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageNumber", value = "页码"),
            @ApiImplicitParam(name = "pageSize", value = "页大小")
    })
    @GetMapping(value = "/files")
    public ResponseEntity<PageResult<FileVo>> getAllFiles(@RequestParam(value = "pageNumber", defaultValue = "0") Integer pageNumber,
                                                          @RequestParam(value = "pageSize", defaultValue = "100") Integer pageSize) throws IOException{
        return new ResponseEntity(new Result().ok("success", fileStoreService.findAllFiles(pageNumber, pageSize)), HttpStatus.OK);
    }


}
