'''
Author: zhuwanjie 2268677665@qq.com
Date: 2025-04-25 17:07:30
LastEditors: zhuwanjie 2268677665@qq.com
LastEditTime: 2025-04-27 09:50:41
FilePath: \egg\cg.py
Description: 这是默认设置,请设置`customMade`, 打开koroFileHeader查看配置 进行设置: https://github.com/OBKoro1/koro1FileHeader/wiki/%E9%85%8D%E7%BD%AE
'''
import numpy as np
import matplotlib.pyplot as plt
import os
from scipy.fftpack import fft2, ifft2


def generate_sampling_mask(shape, sampling_rate):
    """生成随机采样掩码"""
    num_samples = int(sampling_rate * shape[0] * shape[1])
    mask = np.zeros(shape, dtype=np.float32)
    indices = np.random.choice(shape[0] * shape[1], num_samples, replace=False)
    mask.flat[indices] = 1
    return mask


def cg_mri_reconstruction(kspace_sampled, mask, num_iters=200):
    """
    使用共轭梯度法进行MRI重建
    :param kspace_sampled: 采样的k空间数据
    :param mask: 采样掩码
    :param num_iters: 迭代次数
    :return: 重建的图像
    """
    # 初始化重建图像
    img_reconstructed = ifft2(kspace_sampled)

    def A(x):
        # 正向算子：从图像到采样k空间
        kspace = fft2(x)
        return kspace * mask

    def At(y):
        # 伴随算子：从采样k空间到图像
        return ifft2(y)

    # 初始残差
    r = kspace_sampled - A(img_reconstructed)
    p = At(r)

    epsilon = 1e-10  # 一个很小的正数，用于避免除零错误

    for i in range(num_iters):
        Ap = A(p)
        denominator_alpha = np.sum(np.conj(Ap) * Ap)
        if np.abs(denominator_alpha) < epsilon:
            denominator_alpha = epsilon
        alpha = np.sum(np.conj(r) * r) / denominator_alpha
        img_reconstructed = img_reconstructed + alpha * p
        r_new = r - alpha * Ap
        denominator_beta = np.sum(np.conj(r) * r)
        if np.abs(denominator_beta) < epsilon:
            denominator_beta = epsilon
        beta = np.sum(np.conj(r_new) * r_new) / denominator_beta
        p = At(r_new) + beta * p
        r = r_new

    return np.abs(img_reconstructed)

def main():
    # 设置当前工作目录
    current_dir = os.getcwd()

    # 加载完整的k空间数据
    kspace_path = os.path.join(current_dir, 'fid_20250417_103329.npy')
    if not os.path.exists(kspace_path):
        print(f"错误: 找不到文件 {kspace_path}")
        return

    try:
        kspace_full = np.load(kspace_path)
        print(f"成功加载k空间数据: {kspace_path}")
    except Exception as e:
        print(f"错误: 加载k空间数据时出错: {e}")
        return

    # 设置采样率
    sampling_rate = 1  # 可以根据需要调整

    # 生成采样掩码
    mask1 = generate_sampling_mask(kspace_full.shape, sampling_rate)

    # 采样k空间

    # 共轭梯度法重建图像

    # 遍历当前目录下的所有 .npy 文件，排除 'mrd_data.npy'
    for filename in os.listdir(current_dir):
        if filename.lower().endswith('.npy') and filename != 'fid_20250417_103329.npy':
            mask_path = os.path.join(current_dir, filename)
            try:
                mask = np.load(mask_path)
                print(f"处理文件: {mask_path}")
            except Exception as e:
                print(f"错误: 加载文件 {mask_path} 时出错: {e}")
                continue

            # 检查 kspace_full 和 mask 的形状是否匹配
            if kspace_full.shape != mask.shape:
                print(f"警告: 文件 {filename} 的形状与 k 空间数据不匹配，跳过。")
                continue

            # 采样 k 空间
            kspace_sampled = kspace_full * mask1

            # 重建图像
            try:
                img_reconstructed = cg_mri_reconstruction(kspace_sampled, mask1, num_iters=100)
                print(f"成功重建图像: {filename}")
            except Exception as e:
                print(f"错误: 重建文件 {filename} 时出错: {e}")
                continue

            # 构造保存的 PNG 文件名
            base_name = os.path.splitext(filename)[0]
            png_filename = f"{base_name}_cg_reconstructed_fid_20250417_103329.png"
            png_path = os.path.join(current_dir, png_filename)
            #plt.imshow(img_reconstructed)
            plt.imshow(img_reconstructed, cmap='gray')  # 设置为灰度图显示
            plt.axis('off')  # 取消坐标轴
            plt.tight_layout(pad=0)
            plt.savefig(png_path, dpi=300)  # 保存为 PNG 文件


if __name__ == "__main__":
    main()
