import numpy as np
import os
import matplotlib.pyplot as plt
from scipy.fft import fft2, ifft2, fftshift

from scipy.fftpack import dct, idct


def dct2(a):
    """二维离散余弦变换"""
    return dct(dct(a.T, norm='ortho').T, norm='ortho')


def idct2(a):
    """二维离散余弦逆变换"""
    return idct(idct(a.T, norm='ortho').T, norm='ortho')


def fft2c(img):
    """Centered 2D FFT."""
    return np.fft.fftshift(np.fft.fft2(np.fft.ifftshift(img)))


def ifft2c(kspace):
    """Centered 2D IFFT."""
    return np.fft.fftshift(np.fft.ifft2(np.fft.ifftshift(kspace)))


def soft_threshold(x, lam):
    """软阈值操作"""
    return np.sign(x) * np.maximum(np.abs(x) - lam, 0)


def total_variation(x, alpha):
    """计算图像的总变分"""
    dx = np.diff(x, axis=0)
    dy = np.diff(x, axis=1)
    return alpha * (np.sum(np.abs(dx)) + np.sum(np.abs(dy)))


def reconstruct_image(kspace_data):
    im = fftshift(ifft2(kspace_data, (256, 256)), axes=0)
    # 取模得到实值图像
    magnitude_image = np.abs(im)
    # 归一化图像
    normalized_image = (magnitude_image - magnitude_image.min()) / (magnitude_image.max() - magnitude_image.min())
    return normalized_image


def fista_mri_reconstruction(kspace_sampled, mask, lam, num_iters=2000, step_size=0.05, tv_alpha=0.05):
    """
    使用快速迭代软阈值算法FISTA进行MRI重建并引入总变分约束
    :param kspace_sampled: 采样的k空间数据
    :param mask: 采样掩码
    :param lam: 正则化参数
    :param num_iters: 迭代次数
    :param step_size: 步长
    :param tv_alpha: 总变分约束强度
    :return: 重建的图像
    """
    # 对k空间进行频域阈值处理
    kspace_sampled_freq = fft2c(kspace_sampled)
    kspace_sampled_freq_thresholded = soft_threshold(kspace_sampled_freq, 3)
    kspace_sampled_thresholded = ifft2c(kspace_sampled_freq_thresholded)

    # 初始重建（零填充），使用阈值处理后的k空间数据
    img_reconstructed = reconstruct_image(kspace_sampled_thresholded)

    # 显示初始化图像
    plt.imshow(np.abs(img_reconstructed), cmap='gray')
    plt.title('Initial Reconstructed MRI Image')
    plt.show()
    plt.imsave('initial_reconstructed_image.png', np.abs(img_reconstructed), cmap='gray')

    y = img_reconstructed.copy()
    t = 1

    for i in range(num_iters):
        # 将当前图像转换到稀疏域（使用二维离散余弦变换 DCT）
        img_dct = dct2(y)
        img_dct_thresholded = soft_threshold(img_dct, lam * step_size)

        # 将稀疏系数转换回图像域
        img_temp = idct2(img_dct_thresholded)

        # 引入总变分约束
        grad_tv = np.gradient(img_temp)
        grad_tv = np.sqrt(grad_tv[0] ** 2 + grad_tv[1] ** 2)
        img_temp = img_temp - step_size * tv_alpha * grad_tv

        # 数据一致性步骤
        kspace_reconstructed = fft2c(img_temp)
        kspace_reconstructed = kspace_sampled * mask + kspace_reconstructed * (1 - mask)
        img_new = reconstruct_image(kspace_reconstructed)

        t_new = (1 + np.sqrt(1 + 4 * t ** 2)) / 2
        y = img_new + ((t - 1) / t_new) * (img_new - img_reconstructed)

        img_reconstructed = img_new
        t = t_new

        if (i + 1) % 10 == 0 or i == 0:
            print(f"Iteration {i + 1}/{num_iters}")

    return img_reconstructed


def main(kspace_path, mask_path):
    # 设置 FISTA 参数
    lam = 0.05  # 正则化参数，可以根据需要调整
    num_iters = 2000  # 迭代次数
    step_size = 0.05  # 步长，可以根据需要调整
    tv_alpha = 0.1  # 总变分约束强度，可以根据需要调整

    kspace_full = np.load(kspace_path)
    mask = np.load(mask_path)

    # 使用完整k空间数据重建并显示图像
    full_image = reconstruct_image(kspace_full)
    plt.imshow(np.abs(full_image), cmap='gray')
    plt.title('Full MRI Image from Full K-space')
    plt.show()

    # 检查kspace_full和mask的形状是否匹配
    if kspace_full.shape != mask.shape:
        print(f"警告: 文件 {mask_path} 的形状与k空间数据不匹配,程序退出。")
        return

    # 采样k空间
    kspace_sampled = kspace_full * mask

    # 重建图像
    img_reconstructed = fista_mri_reconstruction(kspace_sampled, mask, lam, num_iters, step_size, tv_alpha)

    # 构造保存的PNG文件名
    base_name = os.path.splitext(os.path.basename(mask_path))[0]
    current_dir = os.getcwd()
    png_filename = f"{base_name}_frequency_fista_reconstructed.png"
    png_path = os.path.join(current_dir, png_filename)

    plt.imshow(img_reconstructed, cmap='gray')  # 设置为灰度图显示
    plt.title('Reconstructed MRI Image with Frequency Domain Sparsity')
    plt.show()
    plt.imsave(png_path, np.abs(img_reconstructed), cmap='gray')


if __name__ == "__main__":
    # 在这里直接定义路径
    kspace_path = r'D:\Users\Desktop\egg\fid_20250417_103329.npy'
    mask_path = r'D:\Users\Desktop\egg\radial_mask.npy'
    main(kspace_path, mask_path)
