import numpy as np
import matplotlib.pyplot as plt
from scipy.fftpack import dct, idct
import os


def fft2c(img):
    """Centered 2D FFT."""
    return np.fft.fftshift(np.fft.fft2(np.fft.ifftshift(img)))


def ifft2c(kspace):
    """Centered 2D IFFT."""
    return np.fft.fftshift(np.fft.ifft2(np.fft.ifftshift(kspace)))


def generate_sampling_mask(shape, sampling_rate):
    """生成随机采样掩码"""
    num_samples = int(sampling_rate * shape[0] * shape[1])
    mask = np.zeros(shape, dtype=np.float32)
    indices = np.random.choice(shape[0] * shape[1], num_samples, replace=False)
    mask.flat[indices] = 1
    return mask


def dct2(a):
    """二维离散余弦变换"""
    return dct(dct(a.T, norm='ortho').T, norm='ortho')


def idct2(a):
    """二维离散余弦逆变换"""
    return idct(idct(a.T, norm='ortho').T, norm='ortho')


def cg_mri_reconstruction(kspace_sampled, mask, num_iters=100):
    """
    使用共轭梯度法进行 MRI 重建
    :param kspace_sampled: 采样的 k 空间数据
    :param mask: 采样掩码
    :param num_iters: 迭代次数
    :return: 重建的图像
    """
    # 初始重建（零填充）
    img_reconstructed = ifft2c(kspace_sampled)

    def A_op(x):
        """定义线性算子 A"""
        return fft2c(ifft2c(x) * mask)

    r = kspace_sampled - A_op(img_reconstructed)
    p = r
    for i in range(num_iters):
        Ap = A_op(p)
        alpha = np.sum(np.conj(r) * r) / np.sum(np.conj(p) * Ap)
        img_reconstructed = img_reconstructed + alpha * p
        r_new = r - alpha * Ap
        beta = np.sum(np.conj(r_new) * r_new) / np.sum(np.conj(r) * r)
        p = r_new + beta * p
        r = r_new

        if (i + 1) % 10 == 0 or i == 0:
            print(f"Iteration {i + 1}/{num_iters}")

    return np.abs(img_reconstructed)


def main():
    # 设置当前工作目录
    current_dir = os.getcwd()

    # 加载完整的 k 空间数据
    kspace_path = os.path.join(current_dir, 'mrd_data1.npy')
    if not os.path.exists(kspace_path):
        print(f"错误: 找不到文件 {kspace_path}")
        return

    try:
        kspace_full = np.load(kspace_path)
        print(f"成功加载 k 空间数据: {kspace_path}")
    except Exception as e:
        print(f"错误: 加载 k 空间数据时出错: {e}")
        return

    # 设置共轭梯度法参数
    num_iters = 100  # 迭代次数

    # 遍历当前目录下的所有 .npy 文件，排除 'mrd_data.npy'
    for filename in os.listdir(current_dir):
        if filename.lower().endswith('.npy') and filename != 'mrd_data.npy':
            mask_path = os.path.join(current_dir, filename)
            try:
                mask = np.load(mask_path)
                print(f"处理文件: {mask_path}")
            except Exception as e:
                print(f"错误: 加载文件 {mask_path} 时出错: {e}")
                continue

            # 检查 kspace_full 和 mask 的形状是否匹配
            if kspace_full.shape != mask.shape:
                print(f"警告: 文件 {filename} 的形状与 k 空间数据不匹配，跳过。")
                continue

            # 采样 k 空间
            kspace_sampled = kspace_full * mask

            # 重建图像
            try:
                img_reconstructed = cg_mri_reconstruction(kspace_sampled, mask, num_iters)
                print(f"成功重建图像: {filename}")
            except Exception as e:
                print(f"错误: 使用共轭梯度法重建文件 {filename} 时出错: {e}")
                continue

            # 构造保存的 PNG 文件名
            base_name = os.path.splitext(filename)[0]
            png_filename = f"{base_name}_CMG_reconstructed.png"
            png_path = os.path.join(current_dir, png_filename)
            plt.imshow(img_reconstructed)
            plt.axis('off')  # 取消坐标轴
            plt.tight_layout(pad=0)
            plt.savefig(png_path, dpi=300)  # 保存为 PNG 文件


if __name__ == "__main__":
    main()